<?php
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');

$q = isset($_GET['q']) ? trim($_GET['q']) : '';
$limit = isset($_GET['limit']) ? intval($_GET['limit']) : 50;
if ($limit <= 0) { $limit = 50; }

$dataFile = __DIR__ . '/../../database/indonesia_cities_with_coords.json';
$namesFile = __DIR__ . '/../../database/indonesia_cities_names.json';
if (!file_exists($dataFile)) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Data kota tidak ditemukan di server.'
    ]);
    exit;
}

$json = file_get_contents($dataFile);
$all = json_decode($json, true);
if (!is_array($all)) { $all = []; }

// Load names-only dataset (from SQL) if available and merge for search coverage
$names = [];
if (file_exists($namesFile)) {
    $njson = file_get_contents($namesFile);
    $names = json_decode($njson, true);
    if (!is_array($names)) { $names = []; }
}

$qLower = mb_strtolower($q);
$result = [];
$seen = [];
foreach ($all as $row) {
    $name = isset($row['name']) ? $row['name'] : '';
    $province = isset($row['province']) ? $row['province'] : '';
    $type = isset($row['type']) ? $row['type'] : '';
    $lat = isset($row['lat']) ? floatval($row['lat']) : null;
    $lng = isset($row['lng']) ? floatval($row['lng']) : null;

    if ($qLower === '' || mb_strpos(mb_strtolower($name . ' ' . $province . ' ' . $type), $qLower) !== false) {
        $item = [
            'name' => $name,
            'province' => $province,
            'type' => $type,
            'lat' => $lat,
            'lng' => $lng,
        ];
        $key = strtolower($name.'|'.$province);
        if (!isset($seen[$key])) { $seen[$key]=true; $result[] = $item; }
        if (count($result) >= $limit) break;
    }
}

// Merge names-only if under limit
if (count($result) < $limit && !empty($names)) {
    foreach ($names as $row) {
        $name = isset($row['name']) ? $row['name'] : '';
        $province = isset($row['province']) ? $row['province'] : '';
        $type = isset($row['type']) ? $row['type'] : '';
        $key = strtolower($name.'|'.$province);
        if (isset($seen[$key])) continue;
        if ($qLower === '' || mb_strpos(mb_strtolower($name . ' ' . $province . ' ' . $type), $qLower) !== false) {
            $latlng = resolve_coords($name, $province);
            $item = [
                'name' => $name,
                'province' => $province,
                'type' => $type,
                'lat' => $latlng ? $latlng[0] : null,
                'lng' => $latlng ? $latlng[1] : null,
            ];
            if ($latlng) { append_or_update_coords($dataFile, $item); }
            $result[] = $item;
            $seen[$key] = true;
            if (count($result) >= $limit) break;
        }
    }
}

echo json_encode([
    'success' => true,
    'count' => count($result),
    'data' => $result,
]);
exit;

function resolve_coords($name, $province) {
    $query = trim($name.' '.$province.' Indonesia');
    $url = 'https://nominatim.openstreetmap.org/search?format=json&limit=1&q=' . rawurlencode($query);
    $ctx = stream_context_create([
        'http' => [
            'timeout' => 10,
            'header' => "User-Agent: WarungAjib/1.0\r\nAccept: application/json\r\n"
        ]
    ]);
    $raw = @file_get_contents($url, false, $ctx);
    if (!$raw) return null;
    $arr = json_decode($raw, true);
    if (!is_array($arr) || empty($arr)) return null;
    $lat = isset($arr[0]['lat']) ? floatval($arr[0]['lat']) : null;
    $lon = isset($arr[0]['lon']) ? floatval($arr[0]['lon']) : null;
    if ($lat === null || $lon === null) return null;
    if ($lat < -11 || $lat > 6) return null;
    if ($lon < 95 || $lon > 142) return null;
    return [$lat, $lon];
}

function append_or_update_coords($dataFile, $item) {
    $json = file_exists($dataFile) ? file_get_contents($dataFile) : '[]';
    $data = json_decode($json, true);
    if (!is_array($data)) $data = [];
    $keyNew = strtolower($item['name'].'|'.$item['province']);
    $found = false;
    foreach ($data as &$r) {
        $key = strtolower(($r['name'] ?? '').'|'.($r['province'] ?? ''));
        if ($key === $keyNew) {
            $r['type'] = $item['type'] ?? ($r['type'] ?? '');
            $r['lat'] = $item['lat'];
            $r['lng'] = $item['lng'];
            $found = true; break;
        }
    }
    if (!$found) { $data[] = $item; }
    usort($data, function($a,$b){ $c = strcmp($a['province'],$b['province']); if ($c!==0) return $c; return strcmp($a['name'],$b['name']); });
    file_put_contents($dataFile, json_encode($data, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES));
}
?>
