<?php
require_once '../../cors.php';
require_once '../../config/database.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

$data = json_decode(file_get_contents('php://input'), true);

$orderId = isset($data['order_id']) ? intval($data['order_id']) : 0;
$status = isset($data['status']) ? trim($data['status']) : '';

// Validasi input
if ($orderId <= 0) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Order ID tidak valid']);
    exit;
}

// Status yang diperbolehkan
$allowedStatuses = ['pending', 'processing', 'shipped', 'delivered', 'cancelled'];

if (!in_array($status, $allowedStatuses)) {
    http_response_code(400);
    echo json_encode([
        'success' => false, 
        'message' => 'Status tidak valid. Pilih: pending, processing, shipped, delivered, atau cancelled'
    ]);
    exit;
}

try {
    $database = new Database();
    $db = $database->getConnection();
    
    // Cek apakah order ada
    $checkQuery = "SELECT id, status FROM orders WHERE id = :order_id";
    $checkStmt = $db->prepare($checkQuery);
    $checkStmt->bindParam(':order_id', $orderId);
    $checkStmt->execute();
    
    if ($checkStmt->rowCount() == 0) {
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'Order tidak ditemukan']);
        exit;
    }
    
    $oldStatus = $checkStmt->fetch(PDO::FETCH_ASSOC)['status'];
    
    // Start transaction
    $db->beginTransaction();
    
    try {
        // Update order status
        $query = "UPDATE orders SET status = :status, updated_at = NOW() WHERE id = :order_id";
        $stmt = $db->prepare($query);
        $stmt->bindParam(':status', $status);
        $stmt->bindParam(':order_id', $orderId);
        $stmt->execute();
        
        // Jika order status jadi delivered, update payment status COD jadi verified
        if ($status === 'delivered') {
            $paymentQuery = "UPDATE payments 
                            SET status = 'verified', updated_at = NOW() 
                            WHERE order_id = :order_id 
                            AND payment_method = 'cod' 
                            AND status = 'pending'";
            $paymentStmt = $db->prepare($paymentQuery);
            $paymentStmt->bindParam(':order_id', $orderId);
            $paymentStmt->execute();
        }
        
        // Jika order status jadi cancelled, update payment status COD jadi rejected
        if ($status === 'cancelled') {
            $paymentQuery = "UPDATE payments 
                            SET status = 'rejected', updated_at = NOW() 
                            WHERE order_id = :order_id 
                            AND payment_method = 'cod'";
            $paymentStmt = $db->prepare($paymentQuery);
            $paymentStmt->bindParam(':order_id', $orderId);
            $paymentStmt->execute();
        }
        
        $db->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Status order berhasil diupdate',
            'data' => [
                'order_id' => $orderId,
                'old_status' => $oldStatus,
                'new_status' => $status,
                'updated_at' => date('Y-m-d H:i:s')
            ]
        ]);
        
    } catch (Exception $e) {
        $db->rollBack();
        throw $e;
    }
    
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
?>
