<?php
// Download public GeoJSON of Indonesian kabupaten/kota, compute centroids, export CSV, then import to JSON.
// Usage:
//   php backend_api/tools/fetch_build_cities.php

function stderr($m){ fwrite(STDERR, $m.PHP_EOL); }

$candidates = [
  // Common community datasets (may change over time). Try sequentially until one works.
  'https://raw.githubusercontent.com/superpikar/indonesia-geojson/master/indonesia-kabupaten.geojson',
  'https://raw.githubusercontent.com/superpikar/indonesia-geojson/master/indonesia-kotakab.geojson',
  'https://raw.githubusercontent.com/superpikar/indonesia-geojson/master/indonesia-kabupaten-kota.geojson',
  'https://raw.githubusercontent.com/gismanz/indonesia-geojson/master/kabupaten_kota.geojson',
  'https://raw.githubusercontent.com/aprifan/indonesia-geojson/master/kabupaten_kota.geojson',
];

function http_get($url){
  $ctx = stream_context_create(['http' => ['timeout' => 20]]);
  return @file_get_contents($url, false, $ctx);
}

function centroid_of_coords($coords){
  // $coords: array of [lng, lat]
  if (!$coords || !is_array($coords)) return [null, null];
  $sumLat=0; $sumLng=0; $n=0;
  foreach ($coords as $pt){
    if (!is_array($pt) || count($pt)<2) continue;
    $lng = floatval($pt[0]);
    $lat = floatval($pt[1]);
    if ($lat==0 && $lng==0) continue;
    $sumLat += $lat; $sumLng += $lng; $n++;
  }
  if ($n==0) return [null,null];
  return [$sumLat/$n, $sumLng/$n];
}

function centroid_of_geometry($geom){
  $type = $geom['type'] ?? '';
  $coords = $geom['coordinates'] ?? null;
  if (!$coords) return [null,null];
  if ($type === 'Polygon'){
    // Use outer ring
    $ring = $coords[0] ?? [];
    return centroid_of_coords($ring);
  } elseif ($type === 'MultiPolygon'){
    // Pick first polygon's outer ring
    foreach ($coords as $poly){
      if (isset($poly[0])) return centroid_of_coords($poly[0]);
    }
  }
  // For other types, fallback averaging all points if possible
  return centroid_of_coords(is_array($coords)?$coords:[]);
}

function prop($p, $keys){
  foreach ($keys as $k){
    if (isset($p[$k]) && $p[$k] !== '') return $p[$k];
  }
  return '';
}

$geojson = null; $src = null;
foreach ($candidates as $url){
  stderr("Fetching: $url");
  $raw = http_get($url);
  if (!$raw) continue;
  $json = json_decode($raw, true);
  if (is_array($json) && isset($json['features']) && is_array($json['features'])){
    $geojson = $json; $src = $url; break;
  }
}

if (!$geojson){
  stderr('Failed to download any GeoJSON source. Please check your internet connection or update candidate URLs.');
  exit(1);
}

stderr("Using source: $src");
$features = $geojson['features'];
$rows = [];
$seen = [];
foreach ($features as $f){
  $p = $f['properties'] ?? [];
  $g = $f['geometry'] ?? [];
  list($lat,$lng) = centroid_of_geometry($g);
  if ($lat===null || $lng===null) continue;

  $province = prop($p, ['provinsi','PROVINSI','NAME_1','Propinsi','Provinsi','province','prov_name','prov']) ?: '';
  $name = prop($p, ['kabupaten_kota','KABKOT','WADMKK','WADMKC','Kabupaten','KABUPATEN','Kota','KOTA','NAME_2','name','NAME']) ?: '';
  if ($name==='') continue;

  // Normalize whitespace and casing
  $name = trim($name);
  $province = trim($province);
  // Ensure prefix exists for clarity
  $type = 'Kota';
  $lower = mb_strtolower($name);
  if (strpos($lower, 'kab') === 0 || strpos($lower, 'kabupaten ') === 0){
    $type = 'Kabupaten';
    if (stripos($name, 'Kabupaten') !== 0){ $name = 'Kabupaten '.$name; }
  } elseif (strpos($lower, 'kota') === 0){
    $type = 'Kota';
    if (stripos($name, 'Kota') !== 0){ $name = 'Kota '.$name; }
  } else {
    // Heuristic: treat names containing 'Kota ' as Kota, otherwise Kabupaten if contains 'Kab.' etc.
    if (stripos($name, 'Kota ') === 0) $type = 'Kota';
    elseif (stripos($name, 'Kab ') === 0 || stripos($name, 'Kab.') === 0) { $type = 'Kabupaten'; $name = preg_replace('/^Kab\.?\s*/i','Kabupaten ',$name); }
  }

  $key = strtolower($name.'|'.$province);
  if (isset($seen[$key])) continue; $seen[$key]=true;

  $rows[] = [
    'name' => $name,
    'province' => $province,
    'type' => $type,
    'lat' => round($lat, 6),
    'lng' => round($lng, 6),
  ];
}

// Sort for stability
usort($rows, function($a,$b){
  $c = strcmp($a['province'],$b['province']); if ($c!==0) return $c; return strcmp($a['name'],$b['name']);
});

$csvPath = __DIR__.'/../database/indonesia_cities_with_coords.csv';
$fh = fopen($csvPath,'w');
fputcsv($fh, ['name','province','type','lat','lng']);
foreach ($rows as $r){ fputcsv($fh, $r); }
fclose($fh);

echo "Built CSV with ".count($rows)." rows: $csvPath".PHP_EOL;

// Run importer to generate JSON
$importer = __DIR__.'/import_cities.php';
$php = 'C:\\xampp\\php\\php.exe';
$cmd = escapeshellarg($php).' '.escapeshellarg($importer).' '.escapeshellarg($csvPath);
echo "Running importer: $cmd".PHP_EOL;
system($cmd, $ret);
if ($ret!==0){ stderr('Importer failed.'); exit(1);} else { echo "Importer completed.".PHP_EOL; }
exit(0);
?>
