<?php
// Ambil kabupaten/kota dari API wilayah.id dan isi koordinat untuk dipakai oleh search_cities.php
// Pemakaian:
//   php backend_api/tools/fetch_wilayah_id.php [provCodesCsv] [maxItems]
//   - provCodesCsv opsional: misal "12,32,33" untuk subset. Default: semua kode umum Indonesia.
//   - maxItems opsional: batasi jumlah total kab/kota yang akan di-geocode pada eksekusi ini.

function stderr($m){ fwrite(STDERR, $m.PHP_EOL); }

$defaultCodes = [
  11=>'Aceh',12=>'Sumatera Utara',13=>'Sumatera Barat',14=>'Riau',15=>'Jambi',16=>'Sumatera Selatan',17=>'Bengkulu',18=>'Lampung',19=>'Kepulauan Bangka Belitung',
  21=>'Kepulauan Riau',
  31=>'DKI Jakarta',32=>'Jawa Barat',33=>'Jawa Tengah',34=>'DI Yogyakarta',35=>'Jawa Timur',36=>'Banten',
  51=>'Bali',52=>'Nusa Tenggara Barat',53=>'Nusa Tenggara Timur',
  61=>'Kalimantan Barat',62=>'Kalimantan Tengah',63=>'Kalimantan Selatan',64=>'Kalimantan Timur',65=>'Kalimantan Utara',
  71=>'Sulawesi Utara',72=>'Sulawesi Tengah',73=>'Sulawesi Selatan',74=>'Sulawesi Tenggara',75=>'Gorontalo',76=>'Sulawesi Barat',
  81=>'Maluku',82=>'Maluku Utara',
  91=>'Papua',92=>'Papua Barat'
];

$userCodes = isset($argv[1]) ? trim($argv[1]) : '';
$codes = [];
if ($userCodes !== ''){
  foreach (explode(',', $userCodes) as $c){ $c = intval(trim($c)); if($c>0) $codes[$c] = $defaultCodes[$c] ?? 'Unknown'; }
} else { $codes = $defaultCodes; }

$maxItems = isset($argv[2]) ? intval($argv[2]) : 0; // 0 = tanpa batas

$csvOut = __DIR__.'/../database/indonesia_cities_with_coords.csv';
$jsonOut = __DIR__.'/../database/indonesia_cities_with_coords.json';

$rows = [];
$totalGeocoded = 0;

foreach ($codes as $code => $provName){
  $url = 'https://wilayah.id/api/regencies/'.rawurlencode($code).'.json';
  stderr('Fetch: '.$url);
  $ctx = stream_context_create(['http'=>['timeout'=>20,'header'=>"User-Agent: WarungAjib/1.0\r\nAccept: application/json\r\n"]]);
  $raw = @file_get_contents($url, false, $ctx);
  if (!$raw){ stderr('  Gagal unduh: '.$url); continue; }
  $j = json_decode($raw, true);
  if (!is_array($j) || !isset($j['data']) || !is_array($j['data'])){ stderr('  Format tidak sesuai (butuh data[]).'); continue; }
  foreach ($j['data'] as $item){
    $name = trim($item['name'] ?? '');
    if ($name==='') continue;
    list($normName,$type) = normalize_name($name);
    // Geocode
    $latlng = nominatim($normName.' '.$provName.' Indonesia');
    if ($latlng){ $totalGeocoded++; } else { $latlng = [null,null]; }
    $rows[] = [
      'name' => $normName,
      'province' => $provName,
      'type' => $type,
      'lat' => $latlng[0],
      'lng' => $latlng[1],
    ];
    // Rate limit
    sleep(1);
    if ($maxItems>0 && $totalGeocoded >= $maxItems) break 2;
  }
}

// Dedup & filter: ambil yang punya koordinat terlebih dahulu, tapi tetap simpan yang null
$seen = [];
$final = [];
foreach ($rows as $r){ $k = strtolower($r['name'].'|'.$r['province']); if(isset($seen[$k])) continue; $seen[$k]=true; $final[] = $r; }
usort($final, function($a,$b){ $c = strcmp($a['province'],$b['province']); if($c!==0) return $c; return strcmp($a['name'],$b['name']); });

// Tulis CSV
$fh = fopen($csvOut,'w'); fputcsv($fh,['name','province','type','lat','lng']); foreach($final as $r){ fputcsv($fh,$r);} fclose($fh);
echo 'CSV dibuat: '.$csvOut.' ('.count($final).' baris)'.PHP_EOL;

// Tulis JSON
file_put_contents($jsonOut, json_encode($final, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES));
echo 'JSON dibuat: '.$jsonOut.PHP_EOL;
exit(0);

function normalize_name($name){
  $name = trim($name);
  $type = 'Kota';
  $l = mb_strtolower($name);
  if (strpos($l,'kabupaten')===0){ $type='Kabupaten'; }
  elseif (strpos($l,'kota')===0){ $type='Kota'; }
  else {
    if (stripos($name,'Kab ')===0 || stripos($name,'Kab.')===0){ $type='Kabupaten'; $name=preg_replace('/^Kab\.?\s*/i','Kabupaten ',$name); }
    elseif (stripos($name,'Kota ')===0){ $type='Kota'; }
  }
  // Pastikan awalan eksplisit
  if ($type==='Kabupaten' && stripos($name,'Kabupaten')!==0){ $name='Kabupaten '.$name; }
  if ($type==='Kota' && stripos($name,'Kota')!==0){ $name='Kota '.$name; }
  return [$name,$type];
}

function nominatim($query){
  $url = 'https://nominatim.openstreetmap.org/search?format=json&limit=1&q='.rawurlencode($query);
  $ctx = stream_context_create(['http'=>['timeout'=>15,'header'=>"User-Agent: WarungAjib/1.0\r\nAccept: application/json\r\n"]]);
  $raw = @file_get_contents($url,false,$ctx); if(!$raw) return null;
  $arr = json_decode($raw,true); if(!is_array($arr)||empty($arr)) return null;
  $lat = isset($arr[0]['lat']) ? floatval($arr[0]['lat']) : null;
  $lon = isset($arr[0]['lon']) ? floatval($arr[0]['lon']) : null;
  if ($lat===null || $lon===null) return null;
  if ($lat < -11 || $lat > 6) return null;
  if ($lon < 95 || $lon > 142) return null;
  return [$lat,$lon];
}
?>
