<?php
/**
 * Import Indonesian cities/regencies CSV into JSON used by the API.
 *
 * Usage:
 *   php backend_api/tools/import_cities.php [path/to/indonesia_cities_with_coords.csv]
 *
 * CSV format (header names are case-insensitive):
 *   name, province, type, lat, lng
 * type: Kota | Kabupaten (other values accepted but will be normalized if possible)
 */

function stderr($msg) { fwrite(STDERR, $msg . PHP_EOL); }

$defaultCsv = __DIR__ . '/../database/indonesia_cities_with_coords.csv';
$csvPath = $argv[1] ?? $defaultCsv;
$jsonOut = __DIR__ . '/../database/indonesia_cities_with_coords.json';

if (!file_exists($csvPath)) {
    stderr("CSV not found: $csvPath");
    stderr("Place a CSV at $defaultCsv or pass a path as argument.");
    exit(1);
}

$fh = fopen($csvPath, 'r');
if (!$fh) {
    stderr("Cannot open CSV: $csvPath");
    exit(1);
}

$header = fgetcsv($fh);
if (!$header) {
    stderr('CSV is empty.');
    exit(1);
}

// Map header names to indices
$map = [];
foreach ($header as $i => $h) {
    $key = strtolower(trim($h));
    $map[$key] = $i;
}

$require = ['name','province','type','lat','lng'];
foreach ($require as $k) {
    if (!array_key_exists($k, $map)) {
        stderr("Missing required column: $k");
        exit(1);
    }
}

$rows = [];
$dedup = [];
$line = 1;
while (($cols = fgetcsv($fh)) !== false) {
    $line++;
    $name = trim($cols[$map['name']] ?? '');
    $province = trim($cols[$map['province']] ?? '');
    $type = trim($cols[$map['type']] ?? '');
    $lat = trim($cols[$map['lat']] ?? '');
    $lng = trim($cols[$map['lng']] ?? '');

    if ($name === '' || $province === '' || $lat === '' || $lng === '') {
        continue; // skip incomplete
    }

    // Normalize type
    $typeLower = strtolower($type);
    if (strpos($typeLower, 'kab') === 0) { $type = 'Kabupaten'; }
    else if (strpos($typeLower, 'kota') === 0) { $type = 'Kota'; }
    else if ($type === '') { $type = 'Kota'; }

    $latF = floatval($lat);
    $lngF = floatval($lng);
    if ($latF < -11 || $latF > 6) { continue; }
    if ($lngF < 95 || $lngF > 142) { continue; }

    $key = strtolower($name.'|'.$province);
    if (isset($dedup[$key])) { continue; }
    $dedup[$key] = true;

    $rows[] = [
        'name' => $name,
        'province' => $province,
        'type' => $type,
        'lat' => $latF,
        'lng' => $lngF,
    ];
}

fclose($fh);

// Sort by province then name
usort($rows, function($a,$b){
    $cmp = strcmp($a['province'], $b['province']);
    if ($cmp !== 0) return $cmp;
    return strcmp($a['name'], $b['name']);
});

file_put_contents($jsonOut, json_encode($rows, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES));

echo "Imported ".count($rows)." rows to $jsonOut".PHP_EOL;
exit(0);
