<?php
// Build coordinates dataset from a local GeoJSON file (kabupaten/kota) without internet.
// Usage:
//   php backend_api/tools/local_build_cities.php [path/to/kabupaten_kota.geojson]
// Default path: backend_api/database/kabupaten_kota.geojson

function stderr($m){ fwrite(STDERR, $m.PHP_EOL); }

$default = __DIR__.'/../database/kabupaten_kota.geojson';
$geoPath = $argv[1] ?? $default;

if (!file_exists($geoPath)){
  stderr('GeoJSON tidak ditemukan: '.$geoPath);
  stderr('Letakkan file kabupaten/kota GeoJSON di: '.$default);
  exit(1);
}

$raw = file_get_contents($geoPath);
if (!$raw){ stderr('Gagal membaca GeoJSON.'); exit(1);} 
$json = json_decode($raw, true);
if (!is_array($json) || !isset($json['features']) || !is_array($json['features'])){
  stderr('Format GeoJSON tidak valid (butuh field features[]).');
  exit(1);
}

function centroid_of_coords($coords){
  $sumLat=0; $sumLng=0; $n=0;
  foreach ($coords as $pt){
    if (!is_array($pt) || count($pt)<2) continue;
    $lng = floatval($pt[0]);
    $lat = floatval($pt[1]);
    if ($lat==0 && $lng==0) continue;
    $sumLat += $lat; $sumLng += $lng; $n++;
  }
  if ($n==0) return [null,null];
  return [$sumLat/$n, $sumLng/$n];
}

function centroid_of_geometry($geom){
  $type = $geom['type'] ?? '';
  $coords = $geom['coordinates'] ?? null;
  if (!$coords) return [null,null];
  if ($type === 'Polygon'){
    $ring = $coords[0] ?? [];
    return centroid_of_coords($ring);
  } elseif ($type === 'MultiPolygon'){
    foreach ($coords as $poly){ if (isset($poly[0])) return centroid_of_coords($poly[0]); }
  }
  return centroid_of_coords(is_array($coords)?$coords:[]);
}

function prop($p, $keys){ foreach ($keys as $k){ if (isset($p[$k]) && $p[$k] !== '') return $p[$k]; } return ''; }

function normalize_name($name){
  $name = trim($name);
  $type = 'Kota';
  $lower = mb_strtolower($name);
  if (strpos($lower,'kab')===0 || strpos($lower,'kabupaten ')===0){
    $type='Kabupaten'; if (stripos($name,'Kabupaten')!==0){ $name='Kabupaten '.$name; }
  } elseif (strpos($lower,'kota')===0){
    $type='Kota'; if (stripos($name,'Kota')!==0){ $name='Kota '.$name; }
  } else {
    if (stripos($name,'Kota ')===0) $type='Kota';
    elseif (stripos($name,'Kab ')===0 || stripos($name,'Kab.')===0){ $type='Kabupaten'; $name=preg_replace('/^Kab\.?\s*/i','Kabupaten ',$name); }
  }
  return [$name,$type];
}

$features = $json['features'];
$rows = []; $seen=[];
foreach ($features as $f){
  $p = $f['properties'] ?? [];
  $g = $f['geometry'] ?? [];
  list($lat,$lng) = centroid_of_geometry($g);
  if ($lat===null || $lng===null) continue;
  $province = prop($p, ['provinsi','PROVINSI','NAME_1','Propinsi','Provinsi','province','prov_name','prov']) ?: '';
  $nameRaw = prop($p, ['kabupaten_kota','KABKOT','WADMKK','WADMKC','Kabupaten','KABUPATEN','Kota','KOTA','NAME_2','name','NAME']) ?: '';
  if ($nameRaw==='') continue;
  list($name,$type) = normalize_name($nameRaw);
  $key = strtolower($name.'|'.$province);
  if (isset($seen[$key])) continue; $seen[$key]=true;
  $rows[] = ['name'=>$name,'province'=>trim($province),'type'=>$type,'lat'=>round($lat,6),'lng'=>round($lng,6)];
}

usort($rows, function($a,$b){ $c=strcmp($a['province'],$b['province']); if($c!==0) return $c; return strcmp($a['name'],$b['name']); });

$csvPath = __DIR__.'/../database/indonesia_cities_with_coords.csv';
$fh=fopen($csvPath,'w'); fputcsv($fh,['name','province','type','lat','lng']); foreach($rows as $r){ fputcsv($fh,$r);} fclose($fh);
echo 'CSV dibuat: '.$csvPath.' ('.count($rows).' baris)'.PHP_EOL;

// Jalankan importer JSON
$importer = __DIR__.'/import_cities.php';
$php = 'C:\xampp\php\php.exe';
$cmd = escapeshellarg($php).' '.escapeshellarg($importer).' '.escapeshellarg($csvPath);
echo 'Menjalankan importer: '.$cmd.PHP_EOL;
system($cmd, $ret);
if ($ret!==0){ stderr('Importer gagal.'); exit(1);} else { echo 'Importer selesai.'.PHP_EOL; }
exit(0);
?>
