<?php
session_start();
if (!isset($_SESSION['admin_id'])) {
    header('Location: index.php');
    exit;
}

require_once '../backend_api/config/database.php';
$database = new Database();
$db = $database->getConnection();

// Get all customers (users with role = user)
$query = "SELECT u.*, 
          COUNT(DISTINCT o.id) as total_orders,
          COALESCE(SUM(o.grand_total), 0) as total_spent,
          MAX(o.created_at) as last_order_date
          FROM users u
          LEFT JOIN orders o ON u.id = o.user_id AND o.status != 'cancelled'
          WHERE u.role = 'user'
          GROUP BY u.id
          ORDER BY u.created_at DESC";

$stmt = $db->prepare($query);
$stmt->execute();
$customers = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get statistics
$stats_query = "SELECT 
    COUNT(DISTINCT u.id) as total_customers,
    COUNT(DISTINCT CASE WHEN u.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN u.id END) as new_customers_30d,
    COUNT(DISTINCT CASE WHEN o.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY) THEN u.id END) as active_7d
    FROM users u
    LEFT JOIN orders o ON u.id = o.user_id
    WHERE u.role = 'user'";
$stats_stmt = $db->prepare($stats_query);
$stats_stmt->execute();
$stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);

include 'includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="row mb-4">
        <div class="col-md-12">
            <h2 class="fw-bold gradient-text">
                <i class="fas fa-users me-2"></i>Kelola Pelanggan
            </h2>
            <p class="text-muted">Lihat dan kelola data pelanggan</p>
        </div>
    </div>

    <!-- Statistics Cards -->
    <div class="row mb-4">
        <div class="col-md-4">
            <div class="card border-0 shadow-sm">
                <div class="card-body text-center">
                    <i class="fas fa-users fa-3x text-primary mb-3"></i>
                    <h3 class="fw-bold text-primary"><?= $stats['total_customers'] ?></h3>
                    <p class="text-muted mb-0">Total Pelanggan</p>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="card border-0 shadow-sm">
                <div class="card-body text-center">
                    <i class="fas fa-user-plus fa-3x text-success mb-3"></i>
                    <h3 class="fw-bold text-success"><?= $stats['new_customers_30d'] ?></h3>
                    <p class="text-muted mb-0">Pelanggan Baru (30 Hari)</p>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="card border-0 shadow-sm">
                <div class="card-body text-center">
                    <i class="fas fa-user-check fa-3x text-info mb-3"></i>
                    <h3 class="fw-bold text-info"><?= $stats['active_7d'] ?></h3>
                    <p class="text-muted mb-0">Aktif (7 Hari)</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Customers Table -->
    <div class="row">
        <div class="col-md-12">
            <div class="card border-0 shadow-sm">
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover" id="customersTable">
                            <thead>
                                <tr>
                                    <th>Nama</th>
                                    <th>Email</th>
                                    <th>No. HP</th>
                                    <th>Alamat</th>
                                    <th>Total Pesanan</th>
                                    <th>Total Belanja</th>
                                    <th>Bergabung</th>
                                    <th>Aksi</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($customers as $customer): ?>
                                <tr>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <div class="avatar-circle bg-primary text-white me-2">
                                                <?= strtoupper(substr($customer['nama_lengkap'], 0, 2)) ?>
                                            </div>
                                            <div>
                                                <div class="fw-bold"><?= htmlspecialchars($customer['nama_lengkap']) ?></div>
                                                <small class="text-muted">@<?= htmlspecialchars($customer['username']) ?></small>
                                            </div>
                                        </div>
                                    </td>
                                    <td><?= htmlspecialchars($customer['email']) ?></td>
                                    <td><?= htmlspecialchars($customer['phone'] ?? '-') ?></td>
                                    <td>
                                        <small><?= htmlspecialchars(substr($customer['address'] ?? 'Belum diisi', 0, 50)) ?><?= strlen($customer['address'] ?? '') > 50 ? '...' : '' ?></small>
                                    </td>
                                    <td>
                                        <span class="badge bg-info"><?= $customer['total_orders'] ?> pesanan</span>
                                    </td>
                                    <td>
                                        <strong class="text-success">Rp <?= number_format($customer['total_spent'] ?? 0, 0, ',', '.') ?></strong>
                                    </td>
                                    <td><?= date('d/m/Y', strtotime($customer['created_at'])) ?></td>
                                    <td>
                                        <button class="btn btn-sm btn-info" onclick="viewCustomer(<?= $customer['id'] ?>)">
                                            <i class="fas fa-eye"></i> Detail
                                        </button>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Customer Detail Modal -->
<div class="modal fade" id="customerDetailModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Detail Pelanggan</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="customerDetailContent">
                <div class="text-center py-5">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.avatar-circle {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    font-size: 14px;
}
</style>

<script>
$(document).ready(function() {
    var customersTableId = '#customersTable';
    
    try {
        // Gunakan retrieve: true untuk reuse instance jika sudah ada
        $(customersTableId).DataTable({
            language: {
                url: '//cdn.datatables.net/plug-ins/1.13.7/i18n/id.json'
            },
            order: [[6, 'desc']],
            deferRender: true,
            retrieve: true // Reuse instance jika sudah ada
        });
    } catch (e) {
        console.error('Error initializing DataTable:', e);
    }
});

function viewCustomer(customerId) {
    $('#customerDetailModal').modal('show');
    
    $.ajax({
        url: 'process/get_customer_detail.php',
        method: 'GET',
        data: { id: customerId },
        success: function(response) {
            $('#customerDetailContent').html(response);
        },
        error: function() {
            $('#customerDetailContent').html('<div class="alert alert-danger">Gagal memuat detail pelanggan</div>');
        }
    });
}
</script>

<?php include 'includes/footer.php'; ?>
