<?php
session_start();
if (!isset($_SESSION['admin_id'])) {
    header('Location: index.php');
    exit;
}

require_once '../backend_api/config/database.php';
$database = new Database();
$db = $database->getConnection();

// Get filter status
$filter_status = isset($_GET['status']) ? $_GET['status'] : 'all';

// Build query based on filter
$query = "SELECT o.*, u.nama_lengkap as customer_name, u.phone, 
          p.status as payment_status, p.payment_proof, p.payment_method
          FROM orders o
          LEFT JOIN users u ON o.user_id = u.id
          LEFT JOIN payments p ON o.id = p.order_id
          WHERE 1=1";

if ($filter_status !== 'all') {
    $query .= " AND o.status = :status";
}

$query .= " ORDER BY o.created_at DESC";

$stmt = $db->prepare($query);

if ($filter_status !== 'all') {
    $stmt->bindParam(':status', $filter_status);
}

$stmt->execute();
$orders = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get order statistics
$stats_query = "SELECT 
    COUNT(*) as total,
    SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
    SUM(CASE WHEN status = 'processing' THEN 1 ELSE 0 END) as processing,
    SUM(CASE WHEN status = 'shipped' THEN 1 ELSE 0 END) as shipped,
    SUM(CASE WHEN status = 'delivered' THEN 1 ELSE 0 END) as delivered,
    SUM(CASE WHEN status = 'cancelled' THEN 1 ELSE 0 END) as cancelled
    FROM orders";
$stats_stmt = $db->prepare($stats_query);
$stats_stmt->execute();
$stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);

include 'includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="row mb-4">
        <div class="col-md-12">
            <h2 class="fw-bold gradient-text">
                <i class="fas fa-shopping-cart me-2"></i>Kelola Pesanan
            </h2>
        </div>
    </div>

    <!-- Filter Buttons -->
    <div class="row mb-3">
        <div class="col-md-12">
            <div class="btn-group" role="group">
                <a href="orders.php?status=all" class="btn <?= $filter_status === 'all' ? 'btn-primary' : 'btn-outline-primary' ?>">Semua</a>
                <a href="orders.php?status=pending" class="btn <?= $filter_status === 'pending' ? 'btn-warning' : 'btn-outline-warning' ?>">Pending</a>
                <a href="orders.php?status=processing" class="btn <?= $filter_status === 'processing' ? 'btn-info' : 'btn-outline-info' ?>">Diproses</a>
                <a href="orders.php?status=shipped" class="btn <?= $filter_status === 'shipped' ? 'btn-primary' : 'btn-outline-primary' ?>">Dikirim</a>
                <a href="orders.php?status=delivered" class="btn <?= $filter_status === 'delivered' ? 'btn-success' : 'btn-outline-success' ?>">Selesai</a>
                <a href="orders.php?status=cancelled" class="btn <?= $filter_status === 'cancelled' ? 'btn-danger' : 'btn-outline-danger' ?>">Dibatalkan</a>
            </div>
        </div>
    </div>

    <!-- Orders Table -->
    <div class="row">
        <div class="col-md-12">
            <div class="card border-0 shadow-sm">
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover" id="ordersTable">
                            <thead>
                                <tr>
                                    <th>No. Order</th>
                                    <th>Tanggal</th>
                                    <th>Pelanggan</th>
                                    <th>Total</th>
                                    <th>Metode</th>
                                    <th>Pembayaran</th>
                                    <th>Status</th>
                                    <th>Aksi</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($orders as $order): ?>
                                <tr>
                                    <td><strong>#<?= $order['order_number'] ?></strong></td>
                                    <td><?= date('d/m/Y H:i', strtotime($order['created_at'])) ?></td>
                                    <td>
                                        <div><?= htmlspecialchars($order['customer_name']) ?></div>
                                        <small class="text-muted"><?= htmlspecialchars($order['phone']) ?></small>
                                    </td>
                                    <td><strong>Rp <?= number_format($order['grand_total'], 0, ',', '.') ?></strong></td>
                                    <td>
                                        <?php if ($order['payment_method'] === 'cod'): ?>
                                            <span class="badge bg-info">COD</span>
                                        <?php elseif ($order['payment_method'] === 'qris'): ?>
                                            <span class="badge bg-warning">QRIS</span>
                                        <?php else: ?>
                                            <span class="badge bg-primary">Transfer</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php 
                                        if ($order['payment_method'] === 'cod') {
                                            if ($order['payment_status'] === 'verified') {
                                                echo '<span class="badge bg-success">LUNAS</span>';
                                            } else {
                                                echo '<span class="badge bg-warning">Menunggu Pembayaran</span>';
                                            }
                                        } else {
                                            if ($order['payment_status'] === 'verified') {
                                                echo '<span class="badge bg-success">LUNAS</span>';
                                            } elseif ($order['payment_status'] === 'pending') {
                                                echo '<span class="badge bg-warning">Menunggu</span>';
                                            } else {
                                                echo '<span class="badge bg-danger">Ditolak</span>';
                                            }
                                        }
                                        ?>
                                    </td>
                                    <td>
                                        <?php
                                        $status_badges = [
                                            'pending' => 'warning',
                                            'processing' => 'info',
                                            'shipped' => 'primary',
                                            'delivered' => 'success',
                                            'cancelled' => 'danger'
                                        ];
                                        $status_labels = [
                                            'pending' => 'Pending',
                                            'processing' => 'Diproses',
                                            'shipped' => 'Dikirim',
                                            'delivered' => 'Selesai',
                                            'cancelled' => 'Dibatalkan'
                                        ];
                                        $badge_class = $status_badges[$order['status']] ?? 'secondary';
                                        $status_label = $status_labels[$order['status']] ?? $order['status'];
                                        ?>
                                        <span class="badge bg-<?= $badge_class ?>"><?= $status_label ?></span>
                                    </td>
                                    <td>
                                        <button class="btn btn-sm btn-info" onclick="viewOrder(<?= $order['id'] ?>)">
                                            <i class="fas fa-eye"></i> Detail
                                        </button>
                                        <?php if (($order['payment_method'] === 'transfer' || $order['payment_method'] === 'qris') && $order['payment_proof'] && $order['payment_status'] === 'pending'): ?>
                                        <button class="btn btn-sm btn-success" onclick="verifyPayment(<?= $order['id'] ?>)">
                                            <i class="fas fa-check"></i> Verifikasi
                                        </button>
                                        <?php endif; ?>
                                        <?php if ($order['status'] !== 'cancelled' && $order['status'] !== 'delivered'): ?>
                                        <button class="btn btn-sm btn-primary" onclick="openUpdateStatusModal(<?= $order['id'] ?>, '<?= $order['status'] ?>')">
                                            <i class="fas fa-edit"></i> Ubah Status
                                        </button>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Order Detail Modal -->
<div class="modal fade" id="orderDetailModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Detail Pesanan</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="orderDetailContent">
                <div class="text-center py-5">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Update Status Modal -->
<div class="modal fade" id="updateStatusModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Ubah Status Pesanan</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="updateStatusForm">
                    <input type="hidden" name="order_id" id="statusOrderId">
                    <div class="mb-3">
                        <label class="form-label">Status Saat Ini:</label>
                        <div>
                            <span class="badge bg-secondary fs-6" id="currentStatusBadge">-</span>
                        </div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Ubah Status Ke:</label>
                        <select class="form-select" name="status" id="newStatusSelect" required>
                            <option value="">-- Pilih Status --</option>
                            <option value="pending">Pending</option>
                            <option value="processing">Diproses</option>
                            <option value="shipped">Dikirim</option>
                            <option value="delivered">Selesai</option>
                            <option value="cancelled">Dibatalkan</option>
                        </select>
                    </div>
                    <div class="alert alert-info small">
                        <i class="fas fa-info-circle"></i> 
                        Perubahan status akan langsung terlihat di mobile app user.
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                <button type="button" class="btn btn-primary" id="saveStatusBtn">
                    <i class="fas fa-save"></i> Simpan Perubahan
                </button>
            </div>
        </div>
    </div>
</div>

<?php include 'includes/footer.php'; ?>

<script>
$(document).ready(function() {
    var table = $('#ordersTable').DataTable({
        language: {
            url: '//cdn.datatables.net/plug-ins/1.13.7/i18n/id.json'
        },
        order: [[1, 'desc']]
    });
    
    // Search button click
    $('#btnSearch').on('click', function() {
        var searchText = $('#searchOrder').val();
        table.search(searchText).draw();
    });
    
    // Reset button click
    $('#btnReset').on('click', function() {
        $('#searchOrder').val('');
        $('#filterStatus').val('');
        table.search('').columns().search('').draw();
    });
    
    // Search on enter key
    $('#searchOrder').on('keypress', function(e) {
        if (e.which === 13) {
            var searchText = $(this).val();
            table.search(searchText).draw();
        }
    });
    
    // Filter by status dropdown
    $('#filterStatus').on('change', function() {
        var status = this.value;
        if (status) {
            table.column(5).search(status).draw(); // Column 5 is status
        } else {
            table.column(5).search('').draw();
        }
    });
});

function viewOrder(orderId) {
    $('#orderDetailModal').modal('show');
    
    $.ajax({
        url: 'process/get_order_detail.php',
        method: 'GET',
        data: { id: orderId },
        success: function(response) {
            $('#orderDetailContent').html(response);
        },
        error: function() {
            $('#orderDetailContent').html('<div class="alert alert-danger">Gagal memuat detail pesanan</div>');
        }
    });
}

// Reload detail content in the open modal without reopening it
function loadOrderDetail(orderId) {
    $.ajax({
        url: 'process/get_order_detail.php',
        method: 'GET',
        data: { id: orderId },
        success: function(response) {
            $('#orderDetailContent').html(response);
        },
        error: function() {
            $('#orderDetailContent').html('<div class="alert alert-danger">Gagal memuat detail pesanan</div>');
        }
    });
}

function updateOrderStatus(orderId, newStatus) {
    const statusLabels = {
        'pending': 'Pending',
        'processing': 'Diproses',
        'shipped': 'Dikirim',
        'delivered': 'Selesai',
        'cancelled': 'Dibatalkan'
    };
    
    const confirmMessage = newStatus === 'cancelled' 
        ? 'Apakah Anda yakin ingin MEMBATALKAN pesanan ini?'
        : 'Ubah status pesanan menjadi "' + statusLabels[newStatus] + '"?';
    
    if (!confirm(confirmMessage)) {
        return;
    }
    
    $.ajax({
        url: '../backend_api/api/orders/update_status.php',
        method: 'POST',
        contentType: 'application/json',
        data: JSON.stringify({
            order_id: parseInt(orderId),
            status: newStatus
        }),
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                alert('Status pesanan berhasil diupdate!');
                $('#orderDetailModal').modal('hide');
                location.reload();
            } else {
                alert('Gagal update status: ' + response.message);
            }
        },
        error: function(xhr, status, error) {
            alert('Error: ' + error);
        }
    });
}

function openUpdateStatusModal(orderId, currentStatus) {
    console.log('DEBUG openUpdateStatusModal: orderId =', orderId, 'currentStatus =', currentStatus);
    
    // Close order detail modal first
    $('#orderDetailModal').modal('hide');
    
    // Set values
    $('#statusOrderId').val(orderId);
    $('#newStatusSelect').val(currentStatus);
    
    console.log('DEBUG: Set #statusOrderId to', orderId);
    console.log('DEBUG: #statusOrderId value =', $('#statusOrderId').val());
    
    // Set current status badge
    const statusLabels = {
        'pending': 'Pending',
        'processing': 'Diproses',
        'shipped': 'Dikirim',
        'delivered': 'Selesai',
        'cancelled': 'Dibatalkan'
    };
    const statusColors = {
        'pending': 'secondary',
        'diproses': 'info',
        'dikirim': 'primary',
        'selesai': 'success',
        'dibatalkan': 'danger'
    };
    
    $('#currentStatusBadge').text(statusLabels[currentStatus] || currentStatus)
        .removeClass()
        .addClass('badge bg-' + (statusColors[currentStatus] || 'secondary') + ' fs-6');
    
    // Show modal
    setTimeout(function() {
        $('#updateStatusModal').modal('show');
    }, 300);
}

$('#saveStatusBtn').click(function() {
    const orderId = $('#statusOrderId').val();
    const newStatus = $('#newStatusSelect').val();
    
    console.log('DEBUG: orderId =', orderId);
    console.log('DEBUG: newStatus =', newStatus);
    
    if (!newStatus) {
        alert('Pilih status terlebih dahulu');
        return;
    }
    
    const btn = $(this);
    const originalText = btn.html();
    btn.prop('disabled', true);
    btn.html('<span class="spinner-border spinner-border-sm me-2"></span>Menyimpan...');
    
    const requestData = {
        order_id: parseInt(orderId),
        status: newStatus
    };
    
    console.log('DEBUG: Sending data =', requestData);
    
    $.ajax({
        url: '../backend_api/api/orders/update_status.php',
        method: 'POST',
        contentType: 'application/json',
        data: JSON.stringify(requestData),
        dataType: 'json',
        success: function(response) {
            console.log('DEBUG: Success response =', response);
            btn.prop('disabled', false);
            btn.html(originalText);
            
            if (response.success) {
                alert('Status pesanan berhasil diupdate!');
                $('#updateStatusModal').modal('hide');
                location.reload();
            } else {
                alert('Gagal update status: ' + response.message);
            }
        },
        error: function(xhr, status, error) {
            console.error('DEBUG: Error response =', xhr.responseText);
            console.error('DEBUG: Status =', status);
            console.error('DEBUG: Error =', error);
            btn.prop('disabled', false);
            btn.html(originalText);
            alert('Error: ' + (xhr.responseJSON?.message || xhr.responseText || error));
        }
    });
});

function verifyPayment(orderId) {
    if (confirm('Verifikasi pembayaran pesanan ini?')) {
        $.ajax({
            url: 'process/verify_payment.php',
            method: 'POST',
            data: { order_id: orderId },
            dataType: 'json',
            success: function(response) {
                if (response.success) {
                    alert('Pembayaran berhasil diverifikasi!');
                    location.reload();
                } else {
                    alert('Gagal verifikasi pembayaran: ' + response.message);
                }
            },
            error: function() {
                alert('Terjadi kesalahan saat verifikasi pembayaran');
            }
        });
    }
}
</script>

</body>
</html>
