<?php
session_start();
if (!isset($_SESSION['admin_id'])) {
    header('Location: ../index.php');
    exit;
}

require_once '../../backend_api/config/database.php';
require_once __DIR__ . '/../vendor/setasign/fpdf/fpdf.php';

$database = new Database();
$db = $database->getConnection();

// Get date range from request
$start_date = isset($_GET['start_date']) ? $_GET['start_date'] : date('Y-m-d', strtotime('-30 days'));
$end_date = isset($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-d', strtotime('+1 day'));

// Get data for report
$sales_query = "SELECT 
    DATE(created_at) as date,
    COUNT(*) as total_orders,
    SUM(grand_total) as total_sales
    FROM orders
    WHERE DATE(created_at) BETWEEN :start_date AND :end_date
    AND status != 'cancelled'
    GROUP BY DATE(created_at)
    ORDER BY date DESC";

$sales_stmt = $db->prepare($sales_query);
$sales_stmt->bindParam(':start_date', $start_date);
$sales_stmt->bindParam(':end_date', $end_date);
$sales_stmt->execute();
$sales_data = $sales_stmt->fetchAll(PDO::FETCH_ASSOC);

// Summary Statistics
$summary_query = "SELECT 
    COUNT(*) as total_orders,
    SUM(grand_total) as total_revenue,
    AVG(grand_total) as avg_order_value,
    COUNT(DISTINCT user_id) as unique_customers
    FROM orders
    WHERE DATE(created_at) BETWEEN :start_date AND :end_date
    AND status != 'cancelled'";

$summary_stmt = $db->prepare($summary_query);
$summary_stmt->bindParam(':start_date', $start_date);
$summary_stmt->bindParam(':end_date', $end_date);
$summary_stmt->execute();
$summary = $summary_stmt->fetch(PDO::FETCH_ASSOC);

// Top Products
$products_query = "SELECT 
    p.name,
    p.category,
    SUM(od.quantity) as total_qty,
    SUM(od.subtotal) as total_revenue
    FROM order_details od
    JOIN products p ON od.product_id = p.id
    JOIN orders o ON od.order_id = o.id
    WHERE DATE(o.created_at) BETWEEN :start_date AND :end_date
    AND o.status != 'cancelled'
    GROUP BY od.product_id
    ORDER BY total_revenue DESC
    LIMIT 10";

$products_stmt = $db->prepare($products_query);
$products_stmt->bindParam(':start_date', $start_date);
$products_stmt->bindParam(':end_date', $end_date);
$products_stmt->execute();
$top_products = $products_stmt->fetchAll(PDO::FETCH_ASSOC);

// Top Customers
$customers_query = "SELECT 
    u.nama_lengkap,
    u.email,
    COUNT(o.id) as total_orders,
    SUM(o.grand_total) as total_spent
    FROM users u
    JOIN orders o ON u.id = o.user_id
    WHERE DATE(o.created_at) BETWEEN :start_date AND :end_date
    AND o.status != 'cancelled'
    GROUP BY u.id
    ORDER BY total_spent DESC
    LIMIT 10";

$customers_stmt = $db->prepare($customers_query);
$customers_stmt->bindParam(':start_date', $start_date);
$customers_stmt->bindParam(':end_date', $end_date);
$customers_stmt->execute();
$top_customers = $customers_stmt->fetchAll(PDO::FETCH_ASSOC);

// Create PDF with FPDF
$pdf = new FPDF();
$pdf->AddPage();
$pdf->SetFont('Arial', 'B', 24);
$pdf->SetTextColor(108, 99, 255);
$pdf->Cell(0, 15, 'LAPORAN PENJUALAN', 0, 1, 'C');

$pdf->SetFont('Arial', '', 11);
$pdf->SetTextColor(100, 100, 100);
$pdf->Cell(0, 8, 'Warung Ajib - Analisis Penjualan dan Performa Bisnis', 0, 1, 'C');

// Date Range
$pdf->SetFont('Arial', '', 10);
$pdf->SetTextColor(0, 0, 0);
$pdf->Ln(5);
$pdf->SetFillColor(245, 245, 245);
$pdf->Cell(0, 8, 'Periode: ' . date('d F Y', strtotime($start_date)) . ' - ' . date('d F Y', strtotime($end_date)), 0, 1, 'C', true);

// Summary Section
$pdf->Ln(10);
$pdf->SetFont('Arial', 'B', 12);
$pdf->SetTextColor(108, 99, 255);
$pdf->Cell(0, 8, 'RINGKASAN', 0, 1);

$pdf->SetFont('Arial', '', 10);
$pdf->SetTextColor(0, 0, 0);

// Summary Cards
$col_width = 45;
$pdf->SetFillColor(249, 249, 249);
$summary_items = [
    'Total Pesanan' => number_format($summary['total_orders']),
    'Total Pendapatan' => 'Rp ' . number_format($summary['total_revenue'], 0, ',', '.'),
    'Rata-rata Pesanan' => 'Rp ' . number_format($summary['avg_order_value'], 0, ',', '.'),
    'Pelanggan Unik' => number_format($summary['unique_customers'])
];

foreach ($summary_items as $label => $value) {
    $pdf->SetFont('Arial', 'B', 9);
    $pdf->Cell($col_width - 2, 6, $label, 1, 0, 'C', true);
    $pdf->SetFont('Arial', 'B', 11);
    $pdf->SetTextColor(108, 99, 255);
    $pdf->Cell($col_width - 2, 6, $value, 1, 1, 'C', true);
    $pdf->SetTextColor(0, 0, 0);
}

// Sales Detail Table
$pdf->Ln(8);
$pdf->SetFont('Arial', 'B', 12);
$pdf->SetTextColor(108, 99, 255);
$pdf->Cell(0, 8, 'DETAIL PENJUALAN HARIAN', 0, 1);

$pdf->SetFont('Arial', 'B', 10);
$pdf->SetFillColor(108, 99, 255);
$pdf->SetTextColor(255, 255, 255);
$pdf->Cell(50, 7, 'Tanggal', 1, 0, 'C', true);
$pdf->Cell(35, 7, 'Pesanan', 1, 0, 'C', true);
$pdf->Cell(50, 7, 'Total Penjualan', 1, 0, 'R', true);
$pdf->Cell(40, 7, 'Rata-rata', 1, 1, 'R', true);

$pdf->SetFont('Arial', '', 9);
$pdf->SetTextColor(0, 0, 0);
$fill = false;

foreach ($sales_data as $sale) {
    $avg = $sale['total_sales'] / $sale['total_orders'];
    if ($fill) {
        $pdf->SetFillColor(240, 240, 240);
    } else {
        $pdf->SetFillColor(255, 255, 255);
    }
    
    $date_str = date('d M Y', strtotime($sale['date']));
    $pdf->Cell(50, 6, $date_str, 1, 0, 'L', $fill);
    $pdf->Cell(35, 6, $sale['total_orders'], 1, 0, 'C', $fill);
    $pdf->Cell(50, 6, 'Rp ' . number_format($sale['total_sales'], 0, ',', '.'), 1, 0, 'R', $fill);
    $pdf->Cell(40, 6, 'Rp ' . number_format($avg, 0, ',', '.'), 1, 1, 'R', $fill);
    
    $fill = !$fill;
}

// Top Products Table
$pdf->Ln(8);
$pdf->SetFont('Arial', 'B', 12);
$pdf->SetTextColor(108, 99, 255);
$pdf->Cell(0, 8, 'PRODUK TERLARIS', 0, 1);

$pdf->SetFont('Arial', 'B', 9);
$pdf->SetFillColor(108, 99, 255);
$pdf->SetTextColor(255, 255, 255);
$pdf->Cell(70, 7, 'Produk', 1, 0, 'L', true);
$pdf->Cell(25, 7, 'Kategori', 1, 0, 'C', true);
$pdf->Cell(25, 7, 'Terjual', 1, 0, 'C', true);
$pdf->Cell(45, 7, 'Pendapatan', 1, 1, 'R', true);

$pdf->SetFont('Arial', '', 9);
$pdf->SetTextColor(0, 0, 0);
$fill = false;

foreach ($top_products as $product) {
    if ($fill) {
        $pdf->SetFillColor(240, 240, 240);
    } else {
        $pdf->SetFillColor(255, 255, 255);
    }
    
    $name = substr($product['name'], 0, 25);
    $pdf->Cell(70, 6, $name, 1, 0, 'L', $fill);
    $pdf->Cell(25, 6, substr($product['category'], 0, 12), 1, 0, 'C', $fill);
    $pdf->Cell(25, 6, $product['total_qty'] . ' pcs', 1, 0, 'C', $fill);
    $pdf->Cell(45, 6, 'Rp ' . number_format($product['total_revenue'], 0, ',', '.'), 1, 1, 'R', $fill);
    
    $fill = !$fill;
}

// Top Customers Table
$pdf->Ln(8);
$pdf->SetFont('Arial', 'B', 12);
$pdf->SetTextColor(108, 99, 255);
$pdf->Cell(0, 8, 'PELANGGAN TERBAIK', 0, 1);

$pdf->SetFont('Arial', 'B', 9);
$pdf->SetFillColor(108, 99, 255);
$pdf->SetTextColor(255, 255, 255);
$pdf->Cell(50, 7, 'Pelanggan', 1, 0, 'L', true);
$pdf->Cell(40, 7, 'Email', 1, 0, 'L', true);
$pdf->Cell(25, 7, 'Pesanan', 1, 0, 'C', true);
$pdf->Cell(50, 7, 'Total Belanja', 1, 1, 'R', true);

$pdf->SetFont('Arial', '', 8);
$pdf->SetTextColor(0, 0, 0);
$fill = false;

foreach ($top_customers as $customer) {
    if ($fill) {
        $pdf->SetFillColor(240, 240, 240);
    } else {
        $pdf->SetFillColor(255, 255, 255);
    }
    
    $name = substr($customer['nama_lengkap'], 0, 20);
    $email = substr($customer['email'], 0, 18);
    
    $pdf->Cell(50, 6, $name, 1, 0, 'L', $fill);
    $pdf->Cell(40, 6, $email, 1, 0, 'L', $fill);
    $pdf->Cell(25, 6, $customer['total_orders'] . 'x', 1, 0, 'C', $fill);
    $pdf->Cell(50, 6, 'Rp ' . number_format($customer['total_spent'], 0, ',', '.'), 1, 1, 'R', $fill);
    
    $fill = !$fill;
}

// Footer
$pdf->Ln(10);
$pdf->SetFont('Arial', '', 9);
$pdf->SetTextColor(150, 150, 150);
$pdf->Cell(0, 10, 'Laporan dibuat: ' . date('d F Y H:i:s'), 0, 1, 'C');

// Output PDF as download
header('Content-Type: application/pdf');
header('Content-Disposition: attachment; filename="Laporan_Penjualan_' . date('Y-m-d_H-i-s') . '.pdf"');
header('Cache-Control: no-cache, no-store, must-revalidate');
header('Pragma: no-cache');
header('Expires: 0');

$pdf->Output('D', 'Laporan_Penjualan_' . date('Y-m-d_H-i-s') . '.pdf');
?>
