<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION['admin_id'])) {
    exit('Unauthorized');
}

require_once __DIR__ . '/../../backend_api/config/database.php';
$database = new Database();
$db = $database->getConnection();

$order_id = isset($_GET['id']) ? $_GET['id'] : 0;

$query = "SELECT o.*, u.nama_lengkap, u.email, u.phone, u.address,
          p.payment_proof, p.payment_method, p.status as payment_status
          FROM orders o
          LEFT JOIN users u ON o.user_id = u.id
          LEFT JOIN payments p ON o.id = p.order_id
          WHERE o.id = :order_id";

$stmt = $db->prepare($query);
$stmt->bindParam(':order_id', $order_id);
$stmt->execute();
$order = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$order) {
    echo '<div class="alert alert-danger">Pesanan tidak ditemukan</div>';
    exit;
}

// Get order items
$items_query = "SELECT * FROM order_details WHERE order_id = :order_id";
$items_stmt = $db->prepare($items_query);
$items_stmt->bindParam(':order_id', $order_id);
$items_stmt->execute();
$items = $items_stmt->fetchAll(PDO::FETCH_ASSOC);

$status_badges = [
    'pending' => 'secondary',
    'processing' => 'info',
    'shipped' => 'primary',
    'delivered' => 'success',
    'cancelled' => 'danger'
];

$status_labels = [
    'pending' => 'Pending',
    'processing' => 'Diproses',
    'shipped' => 'Dikirim',
    'delivered' => 'Selesai',
    'cancelled' => 'Dibatalkan'
];

$badge_class = $status_badges[$order['status']] ?? 'secondary';
$status_label = $status_labels[$order['status']] ?? strtoupper($order['status']);
?>

<div class="row">
    <div class="col-md-6">
        <h6 class="fw-bold">Informasi Pesanan</h6>
        <table class="table table-sm">
            <tr>
                <td width="150">No. Pesanan:</td>
                <td><strong>#<?= $order['order_number'] ?></strong></td>
            </tr>
            <tr>
                <td>Tanggal:</td>
                <td><?= date('d F Y H:i', strtotime($order['created_at'])) ?></td>
            </tr>
            <tr>
                <td>Status:</td>
                <td>
                    <span class="badge bg-<?= $badge_class ?>"><?= $status_label ?></span>
                </td>
            </tr>
            <tr>
                <td>Metode Bayar:</td>
                <td>
                    <?php if ($order['payment_method'] === 'cod'): ?>
                        <span class="badge bg-info">COD (Cash on Delivery)</span>
                    <?php elseif ($order['payment_method'] === 'qris'): ?>
                        <span class="badge bg-warning">QRIS</span>
                    <?php else: ?>
                        <span class="badge bg-primary">Transfer Bank</span>
                    <?php endif; ?>
                </td>
            </tr>
            <tr>
                <td>Status Bayar:</td>
                <td>
                    <?php 
                    if ($order['payment_method'] === 'cod') {
                        if ($order['payment_status'] === 'verified') {
                            echo '<span class="badge bg-success">LUNAS</span>';
                        } else {
                            echo '<span class="badge bg-warning">Menunggu Pembayaran</span>';
                        }
                    } elseif ($order['payment_method'] === 'qris' || $order['payment_method'] === 'transfer') {
                        if ($order['payment_status'] === 'verified') {
                            echo '<span class="badge bg-success">LUNAS</span>';
                        } elseif ($order['payment_status'] === 'pending') {
                            echo '<span class="badge bg-warning">Menunggu Verifikasi</span>';
                        } else {
                            echo '<span class="badge bg-danger">Ditolak</span>';
                        }
                    } else {
                        if ($order['payment_status'] === 'verified') {
                            echo '<span class="badge bg-success">LUNAS</span>';
                        } elseif ($order['payment_status'] === 'pending') {
                            echo '<span class="badge bg-warning">Menunggu Verifikasi</span>';
                        } else {
                            echo '<span class="badge bg-danger">Ditolak</span>';
                        }
                    }
                    ?>
                </td>
            </tr>
            <tr>
                <td>Total:</td>
                <td><strong class="text-success">Rp <?= number_format($order['grand_total'], 0, ',', '.') ?></strong></td>
            </tr>
        </table>
    </div>
    <div class="col-md-6">
        <h6 class="fw-bold">Informasi Pelanggan</h6>
        <table class="table table-sm">
            <tr>
                <td width="150">Nama:</td>
                <td><?= htmlspecialchars($order['nama_lengkap']) ?></td>
            </tr>
            <tr>
                <td>Email:</td>
                <td><?= htmlspecialchars($order['email']) ?></td>
            </tr>
            <tr>
                <td>No. HP:</td>
                <td><?= htmlspecialchars($order['phone']) ?></td>
            </tr>
            <tr>
                <td>Alamat:</td>
                <td><?= htmlspecialchars($order['shipping_address']) ?></td>
            </tr>
        </table>
    </div>
</div>

<hr>

<h6 class="fw-bold">Detail Produk</h6>
<div class="table-responsive">
    <table class="table table-sm table-bordered">
        <thead class="table-light">
            <tr>
                <th>Produk</th>
                <th class="text-center">Jumlah</th>
                <th class="text-end">Harga</th>
                <th class="text-end">Subtotal</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($items as $item): ?>
            <tr>
                <td><?= htmlspecialchars($item['product_name']) ?></td>
                <td class="text-center"><?= $item['quantity'] ?></td>
                <td class="text-end">Rp <?= number_format($item['product_price'], 0, ',', '.') ?></td>
                <td class="text-end">Rp <?= number_format($item['subtotal'], 0, ',', '.') ?></td>
            </tr>
            <?php endforeach; ?>
        </tbody>
        <tfoot class="table-light">
            <tr>
                <th colspan="3" class="text-end">Subtotal:</th>
                <th class="text-end">Rp <?= number_format($order['total_price'], 0, ',', '.') ?></th>
            </tr>
            <tr>
                <th colspan="3" class="text-end">Ongkir:</th>
                <th class="text-end">Rp <?= number_format($order['shipping_cost'], 0, ',', '.') ?></th>
            </tr>
            <tr>
                <th colspan="3" class="text-end">Total:</th>
                <th class="text-end text-success">Rp <?= number_format($order['grand_total'], 0, ',', '.') ?></th>
            </tr>
        </tfoot>
    </table>
</div>

<?php if ($order['payment_proof']): ?>
<hr>
<h6 class="fw-bold">Bukti Pembayaran</h6>
<div class="text-center mb-3">
    <?php 
    $proofPath = $order['payment_proof'];
    // Check if path is relative or absolute
    if (strpos($proofPath, 'http') === 0) {
        // It's a full URL
        $imgUrl = $proofPath;
    } else {
        // It's a relative path - check multiple possible locations
        $possiblePaths = [
            '../../uploads/payments/' . $proofPath,
            '../../backend_api/uploads/payments/' . $proofPath,
            'http://192.168.1.7/warung_ajib/backend_api/uploads/payments/' . $proofPath
        ];
        
        $imgUrl = '';
        foreach ($possiblePaths as $path) {
            if (strpos($path, 'http') === 0) {
                $imgUrl = $path;
                break;
            } elseif (file_exists($path)) {
                $imgUrl = str_replace('../../', '../', $path);
                break;
            }
        }
        
        // Fallback to API path
        if (empty($imgUrl)) {
            $imgUrl = 'http://192.168.1.7/warung_ajib/backend_api/uploads/payments/' . basename($proofPath);
        }
    }
    ?>
    <img src="<?= htmlspecialchars($imgUrl) ?>" 
         alt="Bukti Pembayaran" 
         class="img-fluid border rounded shadow-sm" 
         style="max-height: 400px;"
         onerror="this.onerror=null; this.src='data:image/svg+xml,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' width=\'400\' height=\'300\'%3E%3Crect fill=\'%23ddd\' width=\'400\' height=\'300\'/%3E%3Ctext x=\'50%25\' y=\'50%25\' dominant-baseline=\'middle\' text-anchor=\'middle\' font-family=\'Arial\' font-size=\'16\' fill=\'%23999\'%3EGambar tidak ditemukan%3C/text%3E%3C/svg%3E';">
</div>
<div class="row">
    <div class="col-md-12">
        <div class="alert alert-info">
            <strong>Status Pembayaran: </strong>
            <?php 
            if ($order['payment_method'] === 'cod') {
                if ($order['payment_status'] === 'verified') {
                    echo '<span class="badge bg-success fs-6">LUNAS (COD)</span>';
                } else {
                    echo '<span class="badge bg-warning fs-6">Menunggu Pembayaran (COD)</span>';
                }
            } elseif ($order['payment_method'] === 'qris') {
                if ($order['payment_status'] === 'verified') {
                    echo '<span class="badge bg-success fs-6">LUNAS (QRIS)</span>';
                } elseif ($order['payment_status'] === 'pending') {
                    echo '<span class="badge bg-warning fs-6">Menunggu Verifikasi QRIS</span>';
                } else {
                    echo '<span class="badge bg-danger fs-6">Ditolak</span>';
                }
            } else {
                if ($order['payment_status'] === 'verified') {
                    echo '<span class="badge bg-success fs-6">LUNAS</span>';
                } elseif ($order['payment_status'] === 'pending') {
                    echo '<span class="badge bg-warning fs-6">Menunggu Verifikasi</span>';
                } else {
                    echo '<span class="badge bg-danger fs-6">Ditolak</span>';
                }
            }
            ?>
        </div>
        
        <?php if (($order['payment_method'] === 'transfer' || $order['payment_method'] === 'qris') && $order['payment_status'] === 'pending'): ?>
        <div class="d-flex gap-2 justify-content-center">
            <button class="btn btn-success btn-lg" onclick="verifyPaymentAction(<?= $order_id ?>, 'verify')">
                <i class="fas fa-check-circle"></i> Verifikasi Pembayaran
            </button>
            <button class="btn btn-danger btn-lg" onclick="verifyPaymentAction(<?= $order_id ?>, 'reject')">
                <i class="fas fa-times-circle"></i> Tolak & Batalkan Pesanan
            </button>
        </div>
        <?php endif; ?>
    </div>
</div>
<?php endif; ?>

<?php if ($order['notes']): ?>
<hr>
<h6 class="fw-bold">Catatan</h6>
<p><?= htmlspecialchars($order['notes']) ?></p>
<?php endif; ?>

<script>
function verifyPaymentAction(orderId, action) {
    const actionText = action === 'verify' ? 'memverifikasi' : 'menolak';
    const confirmMsg = action === 'verify' 
        ? 'Verifikasi pembayaran pesanan ini?' 
        : 'Tolak pembayaran dan batalkan pesanan ini?';
    
    if (!confirm(confirmMsg)) return;
    
    const btn = event.target;
    const originalText = btn.innerHTML;
    btn.disabled = true;
    btn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Memproses...';
    
    $.ajax({
        url: 'process/verify_payment.php',
        method: 'POST',
        data: { 
            order_id: orderId,
            action: action
        },
        dataType: 'json',
        success: function(response) {
            btn.disabled = false;
            btn.innerHTML = originalText;
            if (response.success) {
                alert('Pembayaran berhasil ' + actionText + '!');
                if (typeof loadOrderDetail === 'function') {
                    loadOrderDetail(orderId);
                } else {
                    location.reload();
                }
            } else {
                alert('Gagal ' + actionText + ' pembayaran: ' + response.message);
            }
        },
        error: function() {
            btn.disabled = false;
            btn.innerHTML = originalText;
            alert('Terjadi kesalahan saat memproses verifikasi');
        }
    });
}
</script>