<?php
session_start();

if (!isset($_SESSION['admin_id'])) {
    header('Location: index.php');
    exit;
}

require_once '../backend_api/config/database.php';

$database = new Database();
$db = $database->getConnection();

// Get all products
$query = "SELECT * FROM products ORDER BY created_at DESC";
$stmt = $db->prepare($query);
$stmt->execute();
$products = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Calculate statistics
$total_products = count($products);
$active_products = count(array_filter($products, function($p) { return $p['is_active']; }));
$low_stock = count(array_filter($products, function($p) { return $p['stock'] < 10; }));
$total_value = array_sum(array_map(function($p) { return $p['price'] * $p['stock']; }, $products));

include 'includes/header.php';
?>

<!-- Page Heading -->
<div class="d-flex justify-content-between align-items-center mb-4">
    <div>
        <h1 class="h3 mb-1 fw-bold text-dark">
            <i class="bi bi-box-seam text-primary me-2"></i>
            Kelola Produk
        </h1>
    </div>
    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addProductModal">
        <i class="bi bi-plus-circle me-2"></i>
        Tambah Produk
    </button>
</div>

<!-- Products Table -->
<div class="card shadow-sm">
    <div class="card-header bg-white py-3">
        <div class="d-flex justify-content-between align-items-center">
            <div class="d-flex gap-2">
                <select class="form-select form-select-sm" id="categoryFilter" style="width: auto;">
                    <option value="">Semua Kategori</option>
                    <option value="JAJANAN">JAJANAN</option>
                    <option value="MAM KENYANG">MAM KENYANG</option>
                    <option value="MINUMAN">MINUMAN</option>
                </select>
            </div>
        </div>
    </div>
    <div class="card-body p-4">
        <div class="table-responsive">
            <table class="table table-hover align-middle mb-0" id="productsTable">
                <thead>
                    <tr>
                        <th class="ps-4">ID</th>
                        <th>Gambar</th>
                        <th>Nama Produk</th>
                        <th>Kategori</th>
                        <th>Harga</th>
                        <th>Stok</th>
                        <th>Terjual</th>
                        <th>Status</th>
                        <th class="pe-4">Aksi</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($products)): ?>
                        <tr>
                            <td colspan="9" class="text-center py-5">
                                <i class="bi bi-inbox fs-1 text-muted d-block mb-3"></i>
                                <p class="text-muted mb-0">Belum ada produk</p>
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($products as $product): ?>
                            <tr>
                                <td class="ps-4">
                                    <span class="text-muted"><?php echo $product['id']; ?></span>
                                </td>
                                <td>
                                    <?php if (!empty($product['image'])): ?>
                                        <img src="../<?php echo $product['image']; ?>" 
                                             alt="<?php echo htmlspecialchars($product['name']); ?>" 
                                             class="rounded"
                                             style="width: 60px; height: 60px; object-fit: cover;">
                                    <?php else: ?>
                                        <div class="rounded bg-light d-flex align-items-center justify-content-center" 
                                             style="width: 60px; height: 60px;">
                                            <i class="bi bi-image text-muted fs-4"></i>
                                        </div>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <div class="fw-semibold"><?php echo htmlspecialchars($product['name']); ?></div>
                                </td>
                                <td>
                                    <span class="badge bg-info bg-opacity-10 text-info">
                                        <?php echo htmlspecialchars($product['category']); ?>
                                    </span>
                                </td>
                                <td>
                                    <span class="fw-semibold">Rp <?php echo number_format($product['price'], 0, ',', '.'); ?></span>
                                </td>
                                <td>
                                    <?php if ($product['stock'] < 10): ?>
                                        <span class="badge bg-warning bg-opacity-10 text-warning">
                                            <i class="bi bi-exclamation-triangle me-1"></i>
                                            <?php echo $product['stock']; ?>
                                        </span>
                                    <?php else: ?>
                                        <span class="badge bg-success bg-opacity-10 text-success">
                                            <?php echo $product['stock']; ?>
                                        </span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <span class="text-muted"><?php echo $product['total_sold'] ?? 0; ?></span>
                                </td>
                                <td>
                                    <?php if ($product['is_active']): ?>
                                        <span class="badge bg-success bg-opacity-10 text-success">
                                            <i class="bi bi-check-circle me-1"></i>Aktif
                                        </span>
                                    <?php else: ?>
                                        <span class="badge bg-secondary bg-opacity-10 text-secondary">
                                            <i class="bi bi-x-circle me-1"></i>Nonaktif
                                        </span>
                                    <?php endif; ?>
                                </td>
                                <td class="pe-4">
                                    <div class="btn-group" role="group">
                                        <button class="btn btn-sm btn-outline-warning" 
                                                onclick='editProduct(<?php echo json_encode($product); ?>)'
                                                title="Edit Produk">
                                            <i class="bi bi-pencil-square"></i>
                                        </button>
                                        <button class="btn btn-sm btn-outline-danger" 
                                                onclick="deleteProduct(<?php echo $product['id']; ?>)"
                                                title="Hapus Produk">
                                            <i class="bi bi-trash"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Add Product Modal -->
<div class="modal fade" id="addProductModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content border-0 shadow-lg">
            <div class="modal-header bg-primary text-white">
                <h5 class="modal-title">
                    <i class="bi bi-plus-circle me-2"></i>
                    Tambah Produk Baru
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <form id="addProductForm">
                <div class="modal-body p-4">
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label fw-semibold">
                                Nama Produk <span class="text-danger">*</span>
                            </label>
                            <input type="text" class="form-control" name="name" 
                                   placeholder="Masukkan nama produk" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label fw-semibold">
                                Kategori <span class="text-danger">*</span>
                            </label>
                            <select class="form-select" name="category" required>
                                <option value="">Pilih Kategori</option>
                                <option value="JAJANAN">JAJANAN</option>
                                <option value="MAM KENYANG">MAM KENYANG</option>
                                <option value="MINUMAN">MINUMAN</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label fw-semibold">
                            Deskripsi <span class="text-danger">*</span>
                        </label>
                        <textarea class="form-control" name="description" rows="3" 
                                  placeholder="Masukkan deskripsi produk" required></textarea>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label fw-semibold">
                                Harga <span class="text-danger">*</span>
                            </label>
                            <div class="input-group">
                                <span class="input-group-text">Rp</span>
                                <input type="number" class="form-control" name="price" 
                                       min="0" placeholder="0" required>
                            </div>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label fw-semibold">
                                Stok <span class="text-danger">*</span>
                            </label>
                            <input type="number" class="form-control" name="stock" 
                                   min="0" placeholder="0" required>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label fw-semibold">Gambar Produk</label>
                        <input type="file" class="form-control" name="image" accept="image/*">
                        <small class="text-muted">
                            <i class="bi bi-info-circle me-1"></i>
                            Format: JPG, PNG, GIF (Maksimal 2MB)
                        </small>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="bi bi-x-circle me-1"></i>
                        Batal
                    </button>
                    <button type="submit" class="btn btn-primary">
                        <i class="bi bi-save me-1"></i>
                        Simpan Produk
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Product Modal -->
<div class="modal fade" id="editProductModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content border-0 shadow-lg">
            <div class="modal-header bg-warning text-dark">
                <h5 class="modal-title">
                    <i class="bi bi-pencil-square me-2"></i>
                    Edit Produk
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="editProductForm">
                <input type="hidden" name="id" id="edit_id">
                <div class="modal-body p-4">
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label fw-semibold">
                                Nama Produk <span class="text-danger">*</span>
                            </label>
                            <input type="text" class="form-control" name="name" id="edit_name" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label fw-semibold">
                                Kategori <span class="text-danger">*</span>
                            </label>
                            <select class="form-select" name="category" id="edit_category" required>
                                <option value="JAJANAN">JAJANAN</option>
                                <option value="MAM KENYANG">MAM KENYANG</option>
                                <option value="MINUMAN">MINUMAN</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label fw-semibold">
                            Deskripsi <span class="text-danger">*</span>
                        </label>
                        <textarea class="form-control" name="description" id="edit_description" rows="3" required></textarea>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label fw-semibold">
                                Harga <span class="text-danger">*</span>
                            </label>
                            <div class="input-group">
                                <span class="input-group-text">Rp</span>
                                <input type="number" class="form-control" name="price" id="edit_price" min="0" required>
                            </div>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label fw-semibold">
                                Stok <span class="text-danger">*</span>
                            </label>
                            <input type="number" class="form-control" name="stock" id="edit_stock" min="0" required>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label fw-semibold">Gambar Produk Baru (Opsional)</label>
                        <input type="file" class="form-control" name="image" accept="image/*">
                        <small class="text-muted">
                            <i class="bi bi-info-circle me-1"></i>
                            Kosongkan jika tidak ingin mengganti gambar
                        </small>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="bi bi-x-circle me-1"></i>
                        Batal
                    </button>
                    <button type="submit" class="btn btn-warning">
                        <i class="bi bi-save me-1"></i>
                        Update Produk
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php include 'includes/footer.php'; ?>

<script>
$(document).ready(function() {
    // Initialize DataTable - cek apakah sudah di-initialize sebelumnya
    var productTableId = '#productsTable';
    var table;
    
    if (!$.fn.DataTable.isDataTable(productTableId)) {
        table = $(productTableId).DataTable({
            language: {
                url: '//cdn.datatables.net/plug-ins/1.13.8/i18n/id.json'
            },
            order: [[0, 'desc']],
            pageLength: 10,
            responsive: true,
            columnDefs: [
                { orderable: false, targets: [1, 8] } // Disable sorting for image and action columns
            ]
        });
    } else {
        table = $(productTableId).DataTable();
    }

    // Category Filter
    $('#categoryFilter').on('change', function() {
        table.column(3).search(this.value).draw();
    });

    // Add Product Form Submit
    $('#addProductForm').on('submit', function(e) {
        e.preventDefault();
        
        var formData = new FormData(this);
        formData.append('action', 'add');
        
        // Disable submit button
        var submitBtn = $(this).find('button[type="submit"]');
        submitBtn.prop('disabled', true).html('<span class="spinner-border spinner-border-sm me-2"></span>Menyimpan...');
        
        $.ajax({
            url: 'process/product_process.php',
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                try {
                    var result = typeof response === 'string' ? JSON.parse(response) : response;
                    if (result.success) {
                        showToast('Produk berhasil ditambahkan!', 'success');
                        $('#addProductModal').modal('hide');
                        setTimeout(function() {
                            location.reload();
                        }, 1000);
                    } else {
                        showToast('Error: ' + result.message, 'danger');
                        submitBtn.prop('disabled', false).html('<i class="bi bi-save me-1"></i>Simpan Produk');
                    }
                } catch (e) {
                    showToast('Terjadi kesalahan: ' + e.message, 'danger');
                    submitBtn.prop('disabled', false).html('<i class="bi bi-save me-1"></i>Simpan Produk');
                }
            },
            error: function(xhr, status, error) {
                showToast('Terjadi kesalahan: ' + error, 'danger');
                submitBtn.prop('disabled', false).html('<i class="bi bi-save me-1"></i>Simpan Produk');
            }
        });
    });

    // Edit Product Form Submit
    $('#editProductForm').on('submit', function(e) {
        e.preventDefault();
        
        var formData = new FormData(this);
        formData.append('action', 'edit');
        
        // Disable submit button
        var submitBtn = $(this).find('button[type="submit"]');
        submitBtn.prop('disabled', true).html('<span class="spinner-border spinner-border-sm me-2"></span>Mengupdate...');
        
        $.ajax({
            url: 'process/product_process.php',
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                try {
                    var result = typeof response === 'string' ? JSON.parse(response) : response;
                    if (result.success) {
                        showToast('Produk berhasil diupdate!', 'success');
                        $('#editProductModal').modal('hide');
                        setTimeout(function() {
                            location.reload();
                        }, 1000);
                    } else {
                        showToast('Error: ' + result.message, 'danger');
                        submitBtn.prop('disabled', false).html('<i class="bi bi-save me-1"></i>Update Produk');
                    }
                } catch (e) {
                    showToast('Terjadi kesalahan: ' + e.message, 'danger');
                    submitBtn.prop('disabled', false).html('<i class="bi bi-save me-1"></i>Update Produk');
                }
            },
            error: function(xhr, status, error) {
                showToast('Terjadi kesalahan: ' + error, 'danger');
                submitBtn.prop('disabled', false).html('<i class="bi bi-save me-1"></i>Update Produk');
            }
        });
    });

    // Reset form when modal is closed
    $('#addProductModal').on('hidden.bs.modal', function() {
        $('#addProductForm')[0].reset();
    });

    $('#editProductModal').on('hidden.bs.modal', function() {
        $('#editProductForm')[0].reset();
    });
});

function editProduct(product) {
    $('#edit_id').val(product.id);
    $('#edit_name').val(product.name);
    $('#edit_description').val(product.description);
    $('#edit_category').val(product.category);
    $('#edit_price').val(product.price);
    $('#edit_stock').val(product.stock);
    $('#editProductModal').modal('show');
}

function deleteProduct(id) {
    if (confirm('Yakin ingin menghapus produk ini?\n\nData yang sudah dihapus tidak dapat dikembalikan.')) {
        $.ajax({
            url: 'process/product_process.php',
            type: 'POST',
            data: {
                action: 'delete',
                id: id
            },
            dataType: 'json',
            success: function(response) {
                if (response.success) {
                    showToast('Produk berhasil dihapus!', 'success');
                    setTimeout(function() {
                        location.reload();
                    }, 1000);
                } else {
                    showToast('Error: ' + response.message, 'danger');
                }
            },
            error: function() {
                showToast('Terjadi kesalahan saat menghapus produk', 'danger');
            }
        });
    }
}
</script>