<?php
session_start();
if (!isset($_SESSION['admin_id'])) {
    header('Location: index.php');
    exit;
}

require_once '../backend_api/config/database.php';
$database = new Database();
$db = $database->getConnection();

// Get date range from filter
// Default to last 30 days to ensure data is visible even with timezone differences
$end_date = isset($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-d', strtotime('+1 day'));
$start_date = isset($_GET['start_date']) ? $_GET['start_date'] : date('Y-m-d', strtotime('-30 days'));

// Sales Report
$sales_query = "SELECT 
    DATE(created_at) as date,
    COUNT(*) as total_orders,
    SUM(grand_total) as total_sales
    FROM orders
    WHERE DATE(created_at) BETWEEN :start_date AND :end_date
    AND status != 'cancelled'
    GROUP BY DATE(created_at)
    ORDER BY date DESC";

$sales_stmt = $db->prepare($sales_query);
$sales_stmt->bindParam(':start_date', $start_date);
$sales_stmt->bindParam(':end_date', $end_date);
$sales_stmt->execute();
$sales_data = $sales_stmt->fetchAll(PDO::FETCH_ASSOC);

// Summary Statistics
$summary_query = "SELECT 
    COUNT(*) as total_orders,
    SUM(grand_total) as total_revenue,
    AVG(grand_total) as avg_order_value,
    COUNT(DISTINCT user_id) as unique_customers
    FROM orders
    WHERE DATE(created_at) BETWEEN :start_date AND :end_date
    AND status != 'cancelled'";

$summary_stmt = $db->prepare($summary_query);
$summary_stmt->bindParam(':start_date', $start_date);
$summary_stmt->bindParam(':end_date', $end_date);
$summary_stmt->execute();
$summary = $summary_stmt->fetch(PDO::FETCH_ASSOC);

// Top Products
$products_query = "SELECT 
    p.name,
    p.category,
    SUM(od.quantity) as total_qty,
    SUM(od.subtotal) as total_revenue
    FROM order_details od
    JOIN products p ON od.product_id = p.id
    JOIN orders o ON od.order_id = o.id
    WHERE DATE(o.created_at) BETWEEN :start_date AND :end_date
    AND o.status != 'cancelled'
    GROUP BY od.product_id
    ORDER BY total_revenue DESC
    LIMIT 10";

$products_stmt = $db->prepare($products_query);
$products_stmt->bindParam(':start_date', $start_date);
$products_stmt->bindParam(':end_date', $end_date);
$products_stmt->execute();
$top_products = $products_stmt->fetchAll(PDO::FETCH_ASSOC);

// Top Customers
$customers_query = "SELECT 
    u.nama_lengkap,
    u.email,
    COUNT(o.id) as total_orders,
    SUM(o.grand_total) as total_spent
    FROM users u
    JOIN orders o ON u.id = o.user_id
    WHERE DATE(o.created_at) BETWEEN :start_date AND :end_date
    AND o.status != 'cancelled'
    GROUP BY u.id
    ORDER BY total_spent DESC
    LIMIT 10";

$customers_stmt = $db->prepare($customers_query);
$customers_stmt->bindParam(':start_date', $start_date);
$customers_stmt->bindParam(':end_date', $end_date);
$customers_stmt->execute();
$top_customers = $customers_stmt->fetchAll(PDO::FETCH_ASSOC);

include 'includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="row mb-4">
        <div class="col-md-8">
            <h2 class="fw-bold gradient-text">
                <i class="fas fa-chart-line me-2"></i>Laporan Penjualan
            </h2>
            <p class="text-muted">Analisis penjualan dan performa bisnis</p>
        </div>
        <div class="col-md-4 text-end">
            <form method="GET" action="process/export_pdf.php" style="display: inline; margin-right: 5px;">
                <input type="hidden" name="start_date" value="<?= $start_date ?>">
                <input type="hidden" name="end_date" value="<?= $end_date ?>">
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-file-pdf me-2"></i>Download PDF
                </button>
            </form>
            <button class="btn btn-outline-primary" onclick="printReport()">
                <i class="fas fa-print me-2"></i>Print
            </button>
        </div>
    </div>

    <!-- Date Filter -->
    <div class="row mb-4">
        <div class="col-md-12">
            <div class="card border-0 shadow-sm">
                <div class="card-body">
                    <form method="GET" action="" class="row g-3">
                        <div class="col-md-4">
                            <label class="form-label">Tanggal Mulai</label>
                            <input type="date" class="form-control" name="start_date" value="<?= $start_date ?>" required>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Tanggal Akhir</label>
                            <input type="date" class="form-control" name="end_date" value="<?= $end_date ?>" required>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">&nbsp;</label>
                            <button type="submit" class="btn btn-primary d-block w-100">
                                <i class="fas fa-filter me-2"></i>Filter
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Summary Cards -->
    <div class="row mb-4">
        <div class="col-md-3">
            <div class="card border-0 shadow-sm">
                <div class="card-body text-center">
                    <i class="fas fa-shopping-cart fa-3x text-primary mb-3"></i>
                    <h3 class="fw-bold text-primary"><?= number_format($summary['total_orders']) ?></h3>
                    <p class="text-muted mb-0">Total Pesanan</p>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card border-0 shadow-sm">
                <div class="card-body text-center">
                    <i class="fas fa-money-bill-wave fa-3x text-success mb-3"></i>
                    <h3 class="fw-bold text-success">Rp <?= number_format($summary['total_revenue'], 0, ',', '.') ?></h3>
                    <p class="text-muted mb-0">Total Pendapatan</p>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card border-0 shadow-sm">
                <div class="card-body text-center">
                    <i class="fas fa-chart-bar fa-3x text-info mb-3"></i>
                    <h3 class="fw-bold text-info">Rp <?= number_format($summary['avg_order_value'], 0, ',', '.') ?></h3>
                    <p class="text-muted mb-0">Rata-rata Pesanan</p>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card border-0 shadow-sm">
                <div class="card-body text-center">
                    <i class="fas fa-users fa-3x text-warning mb-3"></i>
                    <h3 class="fw-bold text-warning"><?= number_format($summary['unique_customers']) ?></h3>
                    <p class="text-muted mb-0">Pelanggan Unik</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Sales Chart -->
    <div class="row mb-4">
        <div class="col-md-12">
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-white">
                    <h5 class="mb-0">Grafik Penjualan Harian</h5>
                </div>
                <div class="card-body">
                    <canvas id="salesChart" height="80"></canvas>
                </div>
            </div>
        </div>
    </div>

    <!-- Top Products & Customers -->
    <div class="row mb-4">
        <div class="col-md-6">
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-white">
                    <h5 class="mb-0"><i class="fas fa-star text-warning me-2"></i>Produk Terlaris</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Produk</th>
                                    <th>Kategori</th>
                                    <th class="text-end">Terjual</th>
                                    <th class="text-end">Pendapatan</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($top_products as $index => $product): ?>
                                <tr>
                                    <td>
                                        <span class="badge bg-primary me-2"><?= $index + 1 ?></span>
                                        <?= htmlspecialchars($product['name']) ?>
                                    </td>
                                    <td><span class="badge bg-info"><?= htmlspecialchars($product['category']) ?></span></td>
                                    <td class="text-end"><?= $product['total_qty'] ?> pcs</td>
                                    <td class="text-end">
                                        <strong class="text-success">Rp <?= number_format($product['total_revenue'], 0, ',', '.') ?></strong>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-md-6">
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-white">
                    <h5 class="mb-0"><i class="fas fa-crown text-warning me-2"></i>Pelanggan Terbaik</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Pelanggan</th>
                                    <th class="text-end">Pesanan</th>
                                    <th class="text-end">Total Belanja</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($top_customers as $index => $customer): ?>
                                <tr>
                                    <td>
                                        <span class="badge bg-warning me-2"><?= $index + 1 ?></span>
                                        <div>
                                            <div><?= htmlspecialchars($customer['nama_lengkap']) ?></div>
                                            <small class="text-muted"><?= htmlspecialchars($customer['email']) ?></small>
                                        </div>
                                    </td>
                                    <td class="text-end"><?= $customer['total_orders'] ?>x</td>
                                    <td class="text-end">
                                        <strong class="text-success">Rp <?= number_format($customer['total_spent'], 0, ',', '.') ?></strong>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Daily Sales Table -->
    <div class="row">
        <div class="col-md-12">
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-white">
                    <h5 class="mb-0">Detail Penjualan Harian</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover" id="salesTable">
                            <thead>
                                <tr>
                                    <th>Tanggal</th>
                                    <th class="text-end">Total Pesanan</th>
                                    <th class="text-end">Total Penjualan</th>
                                    <th class="text-end">Rata-rata</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($sales_data as $sale): ?>
                                <tr>
                                    <td><?= date('l, d F Y', strtotime($sale['date'])) ?></td>
                                    <td class="text-end"><?= $sale['total_orders'] ?> pesanan</td>
                                    <td class="text-end">
                                        <strong class="text-success">Rp <?= number_format($sale['total_sales'], 0, ',', '.') ?></strong>
                                    </td>
                                    <td class="text-end">
                                        Rp <?= number_format($sale['total_sales'] / $sale['total_orders'], 0, ',', '.') ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
$(document).ready(function() {
    var salesTableId = '#salesTable';
    
    try {
        // Gunakan retrieve: true untuk reuse instance jika sudah ada
        $(salesTableId).DataTable({
            language: {
                url: '//cdn.datatables.net/plug-ins/1.13.7/i18n/id.json'
            },
            order: [[0, 'desc']],
            deferRender: true,
            retrieve: true, // Reuse instance jika sudah ada
            drawCallback: function(settings) {
                // Show/hide empty state message
                var api = this.api();
                var rows = api.rows({ search: 'applied' }).count();
                var noDataRow = $(salesTableId + ' tbody tr.no-data-row');
                
                if (rows === 0) {
                    noDataRow.show();
                } else {
                    noDataRow.hide();
                }
            }
    } catch (e) {
        console.error('Error initializing DataTable:', e);
    }

    // Sales Chart
    const salesData = <?= json_encode($sales_data) ?>;
    const dates = salesData.map(item => item.date);
    const sales = salesData.map(item => parseFloat(item.total_sales));

    const ctx = document.getElementById('salesChart').getContext('2d');
    new Chart(ctx, {
        type: 'line',
        data: {
            labels: dates,
            datasets: [{
                label: 'Penjualan (Rp)',
                data: sales,
                borderColor: '#6C63FF',
                backgroundColor: 'rgba(108, 99, 255, 0.1)',
                tension: 0.4,
                fill: true
            }]
        },
        options: {
            responsive: true,
            plugins: {
                legend: {
                    display: true,
                    position: 'top'
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        callback: function(value) {
                            return 'Rp ' + value.toLocaleString('id-ID');
                        }
                    }
                }
            }
        }
    });
});

function printReport() {
    window.print();
}
</script>

<?php include 'includes/footer.php'; ?>
